
/*

	F9 Render Sequence Script
	v1.0
	Matt Gorner

	History:
	
	1.0 (28.09.2009) Initial release

	----------------------------------------------------------------------------------------------------------
	
	Renders a sequence using F9 instead of F10.

	Based on a request by Hieron:
	http://www.newtek.com/forums/showpost.php?p=930623&postcount=362

	Usage:

	1) Set 'Render Display' to Render2Image (to stop the Render Display having lots of frames in its buffer

	2) In the 'Processing' Tab in the 'Effects' panel, add the 'Image Filter' of your choice so your renders are saved ('PSD Export' etc.)
	
	3) Add 'F9 Render Sequence' as a Master Script

	4) Hit the 'Render F9 Sequence' button

*/


@version 2.2
@warnings
@script master
@name "F9 Render Sequence"

@define SCRIPT_NAME    "F9 Render Sequence"
@define SCRIPT_VERSION "1.0"

// Global Variables

rendering;
alertLevel;
first_frame, last_frame, frame_step;
frame_to_render;

gad_first, gad_last, gad_step;

// One-time initialization takes place here

create
{
	// Init variables

    rendering		= false;
    first_frame 	= Scene().renderstart;
    last_frame  	= Scene().renderend;
    frame_step  	= Scene().framestep;
	frame_to_render = first_frame;

	// Open the interface
    options();
}


// Procedure to update script name in Master Plugins to show frame sequence
setDescription
{
	// description = SCRIPT_NAME + " " + first_frame + " / " + last_frame + " Step " + frame_step;
	setdesc( SCRIPT_NAME + " " + first_frame + " / " + last_frame + " Step " + frame_step );
}

flags
{
    // Tell LightWave what type of master script level we are:
    // LWMASTF_LAYOUT	= Layout level script, these aren't removed if the scene is cleared (we also NEED to use this one or the script doesn't work)
	// LWMAST_SCENE		= Scene level script, these will be removed if the scene is cleared

    // Also tell it we want to recieve notifications:
    // LWMASTF_RECEIVE_NOTIFICATIONS

    // These flags definitions can be found in the LW SDK Include file: 'lwmaster.h'

    return(LWMASTF_LAYOUT,LWMASTF_RECEIVE_NOTIFICATIONS);
}

options
{
	// Is the requester open already?  Then close it.
    if(reqisopen())
        reqend();
    else
    {
    	// Otherwise open it up ...
    
	    // User Interface Layout Variables (this is me being anal!) :D

		gad_x				= 10;										// Gadget X coord
		gad_y				= 24;										// Gadget Y coord
		gad_w				= 200;										// Gadget width
		gad_h				= 19;										// Gadget height
		gad_text_offset		= 60;										// Gadget text offset
		num_gads			= 4;										// Total number of gadgets vertically (for calculating the max window height)
		ui_spacing			= 3;										// Spacing gap size
		ui_offset_x 		= 0;										// Main X offset from 0
		ui_offset_y 		= ui_spacing*2;								// Main Y offset from 0
		ui_row				= 0;										// Row number
		ui_row_offset		= gad_h + ui_spacing;						// Row offset
		ui_window_w			= 220;										// Window width
		ui_window_h			= ui_row_offset*num_gads+(2*ui_spacing);	// Window height

		reqbegin(SCRIPT_NAME);
		reqsize(ui_window_w,ui_window_h);

		// Define and layout gadgets ...

    	gad_first = ctlinteger("First Frame", first_frame);
    	ctlposition(gad_first, gad_x, ui_offset_y, gad_w, gad_h, gad_text_offset);
    	ui_offset_y += ui_row_offset;

    	gad_last = ctlinteger("Last Frame", last_frame);
    	ctlposition(gad_last, gad_x, ui_offset_y, gad_w, gad_h, gad_text_offset);
    	ui_offset_y += ui_row_offset;

    	gad_step = ctlinteger("Step", frame_step);
    	ctlposition(gad_step, gad_x, ui_offset_y, gad_w, gad_h, gad_text_offset);
    	ui_offset_y += ui_row_offset;

    	gad_button = ctlbutton("Render F9 Sequence",0,"render_button");
    	ctlposition(gad_button, gad_x, ui_offset_y, gad_w, gad_h, gad_text_offset);
    	ui_offset_y += ui_row_offset;

		// Define gadget refresh procedures when the gadgets are changed

		ctlrefresh( gad_first, "refresh_first");

		ctlrefresh( gad_last, "refresh_last");

		ctlrefresh( gad_step, "refresh_step");

    	reqopen();

		// Update the name in the Master Script window
    	setDescription();
    }
}



// Gadget refresh & callback procedures ...

refresh_first: value
{
	// Update 'first_frame' with the variable (value) passed to the procedure from the gadget
	first_frame = value;

	// Update the name in the Master Script window
	setDescription();
}

refresh_last: value
{
	// Update 'last_frame' with the variable (value) passed to the procedure from the gadget
	last_frame = value;

	// Update the name in the Master Script window
	setDescription();
}

refresh_step: value
{
	// Update 'frame_step' with the variable (value) passed to the procedure from the gadget
	frame_step = value;

	// Update the name in the Master Script window
	setDescription();
}

// Render button callback

render_button
{
    // Suppress dialog boxes
    AutoConfirm(1);

	// Store the current user alert level
    alertLevel = Scene().alertlevel;

    // Set the Alert Level to expert (silence them basically)
    AlertLevel(ALERT_EXPERT);

	// Yes we will be rendering ...
    rendering = true;

    // Got to render procedure
    render();
}



// Render procedure

render
{
	// Go to the frame we want to render (this gets incremented)
	GoToFrame( frame_to_render );

	// Render the frame (equivalent of doing an F9)
    CommandInput("RenderFrame");
}



// Event processing procedure

process: event, command
{

	// We got an event passed, has the render been aborted?
    if(event == LWEVNT_NOTIFY_RENDER_ABORTED)
    {
    	// Apparently so, so update our script variable to reflect we are no longer rendering
        rendering = false;

        // Re-enable dialog boxes and user alert level
        AutoConfirm(0);
        AlertLevel(alertLevel);
    }

	// We got a render complete event (Note: LWEVNT_RENDER_DONE does not work with this script, we need to use this event)
    if(event == LWEVNT_NOTIFY_RENDER_COMPLETE )
    {
    	// Are we still rendering?  And is the frame to render less or equal to the last frame we want to render?
    	if (frame_to_render <= last_frame && rendering == true && (frame_to_render + frame_step < last_frame ))
    	{
    		// Yes, so incrememnt the frame and set the render going again
			frame_to_render+= frame_step;
			render();
		}else
		{
			// Nope, we're finished, reset the variables
			frame_to_render = first_frame;
			rendering = false;

        	// Re-enable dialog boxes and user alert level
            AutoConfirm(0);
            AlertLevel(alertLevel);
		}

    }
}
